<?php
/**
 * --------------------------------------------------
 * KhanBaba Telegram Bot - Core Engine (اصلاح شده)
 * Version: 1.2.0 (Refactored, Optimized, Modern)
 * --------------------------------------------------
 * ✅ دستور !ریست روی دستور ریپلای می‌کند
 * ✅ دستور !شارژ دکمه اشتراک‌گذاری دارد
 * ✅ متن‌ها و آیکون‌های مدرن
 */

final class Khan
{
    private array $message;
    private array $chat;
    private array $from;
    private string $text;
    private int|string $chatId;
    private int $messageId;
    private PDO $db;

    public function __construct(array $message)
    {
        date_default_timezone_set('Asia/Tehran');

        $this->message   = $message;
        $this->chat      = $message['chat'];
        $this->from      = $message['from'];
        $this->text      = trim($message['text'] ?? '');
        $this->chatId    = $this->chat['id'];
        $this->messageId = $message['message_id'];
        $this->db        = Database::instance()->pdo();
    }

    /*--------------------------------------------------
    | Main Processor
    --------------------------------------------------*/
    public function process(): void
    {
        if (!$this->isGroupActive()) {
            $this->handleInactiveGroup();
            return;
        }

        if ($this->isExpired()) {
            $this->handleExpiredGroup();
            return;
        }

        $this->dispatchCommands();
    }

    /*--------------------------------------------------
    | Group State
    --------------------------------------------------*/
    private function isGroupActive(): bool
    {
        $stmt = $this->db->prepare(
            "SELECT is_active FROM groups_data WHERE group_id = ? LIMIT 1"
        );
        $stmt->execute([$this->chatId]);
        return (bool)($stmt->fetchColumn() ?? false);
    }

    private function isExpired(): bool
    {
        $stmt = $this->db->prepare(
            "SELECT expire_date FROM groups_data WHERE group_id = ? AND is_active = 1"
        );
        $stmt->execute([$this->chatId]);
        $expire = $stmt->fetchColumn();

        return $expire && strtotime($expire) < time();
    }

    private function handleInactiveGroup(): void
    {
        if ($this->text === '!Add' && $this->from['id'] === SUDO_ID) {
            $this->handleAddCommand();
            return;
        }

        if (preg_match('/^!گیفت\s+(\S+)$/u', $this->text, $m)) {
            $this->handleGiftCode($m[1]);
        }
    }

    /*--------------------------------------------------
    | Expired Group
    --------------------------------------------------*/
    private function handleExpiredGroup(): void
    {
        deleteGroupData($this->chatId);

        sendMessage(
            $this->chatId,
            "⏰ <b>پایان اعتبار ربات</b>\n\n"
            . "کلیه خدمات ربات در این گروه غیرفعال شد.\n"
            . "برای تمدید اشتراک با مدیریت تماس بگیرید.\n\n"
            . "📌 دستور تمدید:\n<code>!شارژ</code>"
        );

        sleep(2);
        tgRequest('leaveChat', ['chat_id' => $this->chatId]);
    }

    /*--------------------------------------------------
    | Command Dispatcher
    --------------------------------------------------*/
    private function dispatchCommands(): void
    {
        match (true) {
            $this->text === '!Add' && $this->isSudo()          => $this->handleAddCommand(),
            $this->text === '!پیکربندی'                        => $this->handleConfig(),
            $this->text === '!اعتبار'                          => $this->handleCredit(),
            $this->text === '!شارژ' && $this->canCharge()      => $this->handleCharge(),
            $this->text === '!ریست' && $this->isSudo()         => $this->handleReset(),
            $this->text === '!ترک' && $this->isSudo()          => $this->handleLeave(),
            preg_match('/^!گیفت\s+(\S+)$/u', $this->text, $m)  => $this->handleGiftCode($m[1]),
            default                                            => null
        };
    }

    /*--------------------------------------------------
    | Permissions
    --------------------------------------------------*/
    private function isSudo(): bool
    {
        return $this->from['id'] === SUDO_ID;
    }

    private function canCharge(): bool
    {
        return $this->isSudo() || $this->isCreator();
    }

    /*--------------------------------------------------
    | Commands
    --------------------------------------------------*/
    private function handleAddCommand(): void
    {
        sendMessage(
            $this->chatId,
            "💎 <b>پنل نصب ربات خان بابا</b>\n\n"
            . "لطفاً مدت اشتراک مورد نظر را انتخاب کنید:",
            [
                'inline_keyboard' => [
                    [
                        ['text' => '🟢 نصب 7 روزه', 'callback_data' => 'install_7days'],
                        ['text' => '🔵 نصب 30 روزه', 'callback_data' => 'install_30days']
                    ],
                    [
                        ['text' => '❌ بستن پنل', 'callback_data' => 'close_panel']
                    ]
                ]
            ],
            $this->messageId
        );
    }

    private function handleConfig(): void
    {
        $this->syncAdmins();

        $stmt = $this->db->prepare(
            "UPDATE groups_data SET group_title = ? WHERE group_id = ?"
        );
        $stmt->execute([$this->chat['title'], $this->chatId]);

        $members = tgRequest('getChatMemberCount', ['chat_id' => $this->chatId])['result'] ?? 0;
        $admins  = tgRequest('getChatAdministrators', ['chat_id' => $this->chatId])['result'] ?? [];

        $creator = 'نامشخص';
        foreach ($admins as $a) {
            if ($a['status'] === 'creator') {
                $creator = $a['user']['first_name'] ?? 'نامشخص';
                if (!empty($a['user']['username'])) {
                    $creator .= ' (@' . $a['user']['username'] . ')';
                }
                break;
            }
        }

        sendMessage(
            $this->chatId,
            "✅ <b>پیکربندی گروه با موفقیت انجام شد</b>\n\n"
            . "👥 اعضا: {$members}\n"
            . "👨‍💼 مدیران: " . count($admins) . "\n"
            . "👑 سازنده: {$creator}\n\n"
            . "📄 اطلاعات و مدیران بروزرسانی شدند",
            null,
            $this->messageId
        );
    }

    private function handleCredit(): void
    {
        $stmt = $this->db->prepare(
            "SELECT expire_date FROM groups_data WHERE group_id = ?"
        );
        $stmt->execute([$this->chatId]);
        $expire = $stmt->fetchColumn();

        if (!$expire) {
            sendMessage($this->chatId, "❌ اطلاعات اعتبار در دسترس نیست");
            return;
        }

        $remaining = strtotime($expire) - time();
        $days = floor($remaining / 86400);

        $text =
            "📅 <b>وضعیت اعتبار ربات</b>\n\n"
            . "⏳ باقی‌مانده: {$days} روز\n"
            . "📆 انقضا: " . jdate('Y/m/d H:i', strtotime($expire));

        if ($days <= 1) {
            $text .= "\n\n⚠️ <b>هشدار:</b> اعتبار رو به پایان است";
            $this->warnExpiry();
        }

        sendMessage($this->chatId, $text, null, $this->messageId);
    }

    private function handleCharge(): void
    {
        // دریافت لینک گروه برای اشتراک‌گذاری
        $groupLink = getGroupInviteLink($this->chatId);
        $shareText = urlencode("✨ درخواست تمدید اشتراک ربات\n\n👥 گروه: {$this->chat['title']}\n\n🔗 لینک گروه:");
        
        $text = "📨 <b>درخواست تمدید اشتراک</b>\n\n";
        $text .= "👥 <b>گروه:</b> {$this->chat['title']}\n";
        $text .= "🆔 <b>شناسه:</b> <code>{$this->chatId}</code>\n";
        $text .= "👤 <b>درخواست‌کننده:</b> " . getUserMention($this->from);

        // دکمه اشتراک‌گذاری لینک گروه
        $keyboard = null;
        if ($groupLink) {
            $shareUrl = "https://t.me/share/url?url=" . urlencode($groupLink) . "&text=" . $shareText;
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => '📤 اشتراک‌گذاری لینک گروه', 'url' => $shareUrl]],
                    [['text' => '🔗 ورود به گروه', 'url' => $groupLink]]
                ]
            ];
        }

        sendMessage(SUDO_ID, $text, $keyboard);

        sendMessage(
            $this->chatId,
            "✅ درخواست تمدید برای مدیریت ارسال شد.\n"
            . "لطفاً منتظر بررسی باشید.",
            null,
            $this->messageId
        );
    }

    private function handleReset(): void
    {
        $this->db
            ->prepare("UPDATE groups_data SET is_active = 0 WHERE group_id = ?")
            ->execute([$this->chatId]);

        // ✅ ریپلای روی دستور
        sendMessage(
            $this->chatId, 
            "♻️ <b>تنظیمات گروه ریست شد</b>", 
            null, 
            $this->messageId
        );
    }

    private function handleLeave(): void
    {
        deleteGroupData($this->chatId);

        sendMessage(
            $this->chatId,
            "👋 <b>ربات از گروه خارج می‌شود</b>\n\n"
            . "تمام اطلاعات گروه حذف شد"
        );

        sleep(2);
        tgRequest('leaveChat', ['chat_id' => $this->chatId]);
    }

    private function handleGiftCode(string $code): void
    {
        $stmt = $this->db->prepare(
            "SELECT * FROM gift_codes WHERE code = ? AND is_used = 0"
        );
        $stmt->execute([$code]);
        $gift = $stmt->fetch();

        if (!$gift) {
            sendMessage($this->chatId, "❌ کد گیفت نامعتبر یا استفاده‌شده است", null, $this->messageId);
            return;
        }

        // فعال‌سازی گروه (بدون ارسال پیام اضافی)
        $expireDate = date('Y-m-d H:i:s', strtotime("+{$gift['days']} days"));
        
        $stmt = $this->db->prepare("
            INSERT INTO groups_data (group_id, group_title, is_active, expire_date, updated_at)
            VALUES (?, ?, 1, ?, NOW())
            ON DUPLICATE KEY UPDATE
                is_active = 1,
                expire_date = VALUES(expire_date),
                group_title = VALUES(group_title),
                updated_at = NOW()
        ");
        $stmt->execute([$this->chatId, $this->chat['title'], $expireDate]);

        // علامت‌گذاری کد به عنوان استفاده شده
        $this->db->prepare(
            "UPDATE gift_codes SET is_used = 1, used_by_group = ?, used_at = NOW() WHERE code = ?"
        )->execute([$this->chatId, $code]);

        // ✅ فقط یک پیام ریپلای روی دستور
        sendMessage(
            $this->chatId,
            "🎉 <b>اشتراک با موفقیت فعال شد!</b>\n\n"
            . "⏰ <b>مدت اعتبار:</b> {$gift['days']} روز\n"
            . "📅 <b>انقضا:</b> " . jdate('Y/m/d', strtotime($expireDate)) . "\n\n"
            . "✨ تمامی امکانات ربات فعال است!",
            null,
            $this->messageId
        );
    }

    /*--------------------------------------------------
    | Admin Utilities
    --------------------------------------------------*/
    private function syncAdmins(): void
    {
        $admins = tgRequest('getChatAdministrators', ['chat_id' => $this->chatId])['result'] ?? [];

        $this->db->prepare(
            "DELETE FROM group_admins WHERE group_id = ?"
        )->execute([$this->chatId]);

        foreach ($admins as $a) {
            $u = $a['user'];
            $this->db->prepare(
                "INSERT INTO group_admins 
                (group_id, user_id, username, first_name, is_creator)
                VALUES (?, ?, ?, ?, ?)"
            )->execute([
                $this->chatId,
                $u['id'],
                $u['username'] ?? null,
                $u['first_name'] ?? null,
                $a['status'] === 'creator' ? 1 : 0
            ]);
        }
    }

    private function isCreator(): bool
    {
        $stmt = $this->db->prepare(
            "SELECT 1 FROM group_admins 
             WHERE group_id = ? AND user_id = ? AND is_creator = 1"
        );
        $stmt->execute([$this->chatId, $this->from['id']]);
        return (bool)$stmt->fetchColumn();
    }

    private function warnExpiry(): void
    {
        $msg = sendMessage(
            $this->chatId,
            "⚠️ <b>هشدار تمدید</b>\n\n"
            . "اعتبار ربات رو به پایان است.\n"
            . "برای تمدید: <code>!شارژ</code>"
        );

        if (!empty($msg['result']['message_id'])) {
            tgRequest('pinChatMessage', [
                'chat_id' => $this->chatId,
                'message_id' => $msg['result']['message_id']
            ]);
        }
    }
}