<?php
/**
 * سیستم کامل مدیریت مقام‌ها - نسخه کامل
 * MIR Management System - Full Version with Panel
 */

require_once 'config.php';
require_once 'jdate.php';

class MirSystem {
    private $db;
    private $message;
    private $chat_id;
    private $from_id;
    private $message_id;
    private $text;
    
    public function __construct($message) {
        $this->db = Database::instance()->pdo();
        $this->message = $message;
        $this->chat_id = $message['chat']['id'];
        $this->from_id = $message['from']['id'];
        $this->message_id = $message['message_id'];
        $this->text = trim($message['text'] ?? '');
    }
    
    /**
     * پردازش دستورات
     */
    public function processCommand() {
        if (!$this->isGroupActive()) {
            return false;
        }
        
        $text = $this->text;
        
        // پنل کاربر - با یا بدون فاصله
        if ($text == 'پنل کاربر' || preg_match('/^پنل\s*کاربر\s*(@?\w+|\d+)$/u', $text)) {
            return $this->showUserPanel();
        }
        
        // دستورات مالک (وزیر اعظم) - با یا بدون فاصله
        if ($text == 'تنظیم مالک' || $text == 'ارتقا مالک' || preg_match('/^(تنظیم|ارتقا)\s*مالک\s*(@?\w+|\d+)$/u', $text)) {
            return $this->setOwner();
        }
        if ($text == 'حذف مالک' || $text == 'عزل مالک' || preg_match('/^(حذف|عزل)\s*مالک\s*(@?\w+|\d+)$/u', $text)) {
            return $this->removeOwner();
        }
        if ($text == 'لیست مالکین' || $text == 'لیست مالک') {
            return $this->listOwners();
        }
        
        // دستورات معاون - با یا بدون فاصله
        if ($text == 'تنظیم معاون' || $text == 'ارتقا معاون' || preg_match('/^(تنظیم|ارتقا)\s*معاون\s*(@?\w+|\d+)$/u', $text)) {
            return $this->setDeputy();
        }
        if ($text == 'حذف معاون' || $text == 'عزل معاون' || preg_match('/^(حذف|عزل)\s*معاون\s*(@?\w+|\d+)$/u', $text)) {
            return $this->removeDeputy();
        }
        if ($text == 'لیست معاون' || $text == 'لیست معاونین') {
            return $this->listDeputies();
        }
        
        // دستورات مشاور - با یا بدون فاصله
        if ($text == 'تنظیم مشاور' || $text == 'ارتقا مشاور' || preg_match('/^(تنظیم|ارتقا)\s*مشاور\s*(@?\w+|\d+)$/u', $text)) {
            return $this->setAdvisor();
        }
        if ($text == 'حذف مشاور' || $text == 'عزل مشاور' || preg_match('/^(حذف|عزل)\s*مشاور\s*(@?\w+|\d+)$/u', $text)) {
            return $this->removeAdvisor();
        }
        if ($text == 'لیست مشاور' || $text == 'لیست مشاورین') {
            return $this->listAdvisors();
        }
        
        // دستورات مدیر - با یا بدون فاصله
        if ($text == 'تنظیم مدیر' || $text == 'ارتقا مدیر' || preg_match('/^(تنظیم|ارتقا)\s*مدیر\s*(@?\w+|\d+)$/u', $text)) {
            return $this->setAdmin();
        }
        if ($text == 'حذف مدیر' || $text == 'عزل مدیر' || preg_match('/^(حذف|عزل)\s*مدیر\s*(@?\w+|\d+)$/u', $text)) {
            return $this->removeAdmin();
        }
        if ($text == 'لیست مدیران' || $text == 'لیست مدیر') {
            return $this->listAdmins();
        }
        
        // دستورات ویژه - با یا بدون فاصله
        if ($text == 'تنظیم ویژه' || $text == 'ارتقا ویژه' || preg_match('/^(تنظیم|ارتقا)\s*ویژه\s*(@?\w+|\d+)$/u', $text)) {
            return $this->setVip();
        }
        if ($text == 'حذف ویژه' || $text == 'عزل ویژه' || preg_match('/^(حذف|عزل)\s*ویژه\s*(@?\w+|\d+)$/u', $text)) {
            return $this->removeVip();
        }
        if ($text == 'لیست ویژه') {
            return $this->listVips();
        }
        
        // دستورات بن - با یا بدون فاصله
        if ($text == 'بن' || preg_match('/^بن\s*(@?\w+|\d+)$/u', $text)) {
            return $this->banUser();
        }
        if ($text == 'حذف بن' || preg_match('/^حذف\s*بن\s*(@?\w+|\d+)$/u', $text)) {
            return $this->unbanUser();
        }
        if ($text == 'لیست بن') {
            return $this->listBanned();
        }
        
        // بن سراسری (فقط سودو) - با یا بدون فاصله
        if ($text == 'بن+' || $text == '!بن' || preg_match('/^(بن\+|!بن)\s*(@?\w+|\d+)$/u', $text)) {
            return $this->globalBan();
        }
        if ($text == '!حذف بن' || preg_match('/^!حذف\s*بن\s*(@?\w+|\d+)$/u', $text)) {
            return $this->removeGlobalBan();
        }
        
        // دستورات سکوت - با یا بدون فاصله
        if ($text == 'سکوت' || preg_match('/^سکوت\s*(@?\w+|\d+)$/u', $text)) {
            return $this->muteUser();
        }
        if ($text == 'حذف سکوت' || preg_match('/^حذف\s*سکوت\s*(@?\w+|\d+)$/u', $text)) {
            return $this->unmuteUser();
        }
        if ($text == 'لیست سکوت') {
            return $this->listMuted();
        }
        
        // دستورات اخطار - با یا بدون فاصله
        if ($text == 'اخطار' || preg_match('/^اخطار\s*(@?\w+|\d+)$/u', $text)) {
            return $this->warnUser();
        }
        if ($text == 'حذف اخطار' || preg_match('/^حذف\s*اخطار\s*(@?\w+|\d+)$/u', $text)) {
            return $this->removeWarn();
        }
        if ($text == 'حذف اخطارها' || preg_match('/^حذف\s*اخطارها\s*(@?\w+|\d+)$/u', $text)) {
            return $this->clearWarns();
        }
        if (preg_match('/^تعداد اخطار (\d+)$/u', $text, $matches)) {
            return $this->setWarnLimit($matches[1]);
        }
        if ($text == 'لیست اخطار') {
            return $this->listWarned();
        }
        
        return false;
    }
    
    /**
     * نمایش پنل کاربر
     */
    private function showUserPanel() {
        if (!$this->checkPanelPermission()) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        $permissions = $this->getUserPermissions($target['id']);
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    [
                        'text' => ($permissions['can_ban'] ? '✅' : '❌') . ' بن',
                        'callback_data' => 'panel_toggle_ban_' . $target['id']
                    ],
                    [
                        'text' => ($permissions['can_mute'] ? '✅' : '🔇') . ' سکوت',
                        'callback_data' => 'panel_toggle_mute_' . $target['id']
                    ]
                ],
                [
                    [
                        'text' => ($permissions['can_delete'] ? '✅' : '🗑') . ' حذف پیام',
                        'callback_data' => 'panel_toggle_delete_' . $target['id']
                    ],
                    [
                        'text' => ($permissions['can_pin'] ? '✅' : '📌') . ' سنجاق',
                        'callback_data' => 'panel_toggle_pin_' . $target['id']
                    ]
                ],
                [
                    [
                        'text' => '💾 تایید و ذخیره',
                        'callback_data' => 'panel_save_' . $target['id']
                    ]
                ],
                [
                    [
                        'text' => '❌ بستن پنل',
                        'callback_data' => 'panel_close'
                    ]
                ]
            ]
        ];
        
        $mention = $this->getUserMention($target);
        $rank = $this->getUserRank($target['id']);
        
        $text = "👤 <b>پنل مدیریت کاربر</b>\n\n";
        $text .= "👨‍💼 <b>کاربر:</b> {$mention}\n";
        $text .= "🆔 <b>آیدی:</b> <code>{$target['id']}</code>\n";
        $text .= "👑 <b>مقام فعلی:</b> {$rank}\n\n";
        $text .= "━━━━━━━━━━━━━━━━\n\n";
        $text .= "🔹 <b>دسترسی‌های فعلی:</b>\n";
        $text .= "• بن: " . ($permissions['can_ban'] ? "✅ فعال" : "❌ غیرفعال") . "\n";
        $text .= "• سکوت: " . ($permissions['can_mute'] ? "✅ فعال" : "❌ غیرفعال") . "\n";
        $text .= "• حذف پیام: " . ($permissions['can_delete'] ? "✅ فعال" : "❌ غیرفعال") . "\n";
        $text .= "• سنجاق: " . ($permissions['can_pin'] ? "✅ فعال" : "❌ غیرفعال") . "\n\n";
        $text .= "💡 <b>راهنما:</b>\n";
        $text .= "• کلیک روی هر دکمه → تغییر وضعیت\n";
        $text .= "• ✅ = فعال | ❌ = غیرفعال\n";
        $text .= "• برای اعمال تغییرات روی «تایید و ذخیره» کلیک کنید";
        
        $this->saveTempPermissions($target['id'], $permissions);
        
        sendMessage($this->chat_id, $text, $keyboard, $this->message_id);
        return true;
    }
    
    /**
     * مدیریت callback های پنل
     */
    public function handlePanelCallback($callback_data, $callback_id, $message_id, $from_id) {
        if (!$this->checkPanelPermission($from_id)) {
            sendRequest('answerCallbackQuery', [
                'callback_query_id' => $callback_id,
                'text' => '⛔️ شما دسترسی ندارید!',
                'show_alert' => true
            ]);
            return;
        }
        
        $parts = explode('_', $callback_data);
        $action = $parts[1] ?? '';
        
        if ($action == 'toggle') {
            $permission_type = $parts[2] ?? '';
            $user_id = $parts[3] ?? 0;
            
            $permissions = $this->getTempPermissions($user_id);
            
            switch ($permission_type) {
                case 'ban':
                    $permissions['can_ban'] = !$permissions['can_ban'];
                    break;
                case 'mute':
                    $permissions['can_mute'] = !$permissions['can_mute'];
                    break;
                case 'delete':
                    $permissions['can_delete'] = !$permissions['can_delete'];
                    break;
                case 'pin':
                    $permissions['can_pin'] = !$permissions['can_pin'];
                    break;
            }
            
            $this->saveTempPermissions($user_id, $permissions);
            $this->updatePanelMessage($message_id, $user_id, $permissions);
            
            sendRequest('answerCallbackQuery', [
                'callback_query_id' => $callback_id,
                'text' => '✅ تغییر اعمال شد'
            ]);
            
        } elseif ($action == 'save') {
            $user_id = $parts[2] ?? 0;
            $permissions = $this->getTempPermissions($user_id);
            
            $this->saveUserPermissions($user_id, $permissions);
            $this->clearTempPermissions($user_id);
            $this->updateUserRank($user_id, $permissions);
            
            // پیام ساده بدون جزئیات
            $text = "✅ <b>تنظیمات با موفقیت ذخیره شد</b>";
            
            editMessage($this->chat_id, $message_id, $text);
            
            sendRequest('answerCallbackQuery', [
                'callback_query_id' => $callback_id,
                'text' => '💾 ذخیره شد!',
                'show_alert' => true
            ]);
            
        } elseif ($action == 'close') {
            // بستن پنل - پیام ساده
            $text = "❌ <b>پنل بسته شد</b>";
            editMessage($this->chat_id, $message_id, $text);
            
            sendRequest('answerCallbackQuery', [
                'callback_query_id' => $callback_id,
                'text' => 'پنل بسته شد'
            ]);
        }
    }
    
    /**
     * بروزرسانی پیام پنل
     */
    private function updatePanelMessage($message_id, $user_id, $permissions) {
        $keyboard = [
            'inline_keyboard' => [
                [
                    [
                        'text' => ($permissions['can_ban'] ? '✅' : '❌') . ' بن',
                        'callback_data' => 'panel_toggle_ban_' . $user_id
                    ],
                    [
                        'text' => ($permissions['can_mute'] ? '✅' : '🔇') . ' سکوت',
                        'callback_data' => 'panel_toggle_mute_' . $user_id
                    ]
                ],
                [
                    [
                        'text' => ($permissions['can_delete'] ? '✅' : '🗑') . ' حذف پیام',
                        'callback_data' => 'panel_toggle_delete_' . $user_id
                    ],
                    [
                        'text' => ($permissions['can_pin'] ? '✅' : '📌') . ' سنجاق',
                        'callback_data' => 'panel_toggle_pin_' . $user_id
                    ]
                ],
                [
                    [
                        'text' => '💾 تایید و ذخیره',
                        'callback_data' => 'panel_save_' . $user_id
                    ]
                ],
                [
                    [
                        'text' => '❌ بستن پنل',
                        'callback_data' => 'panel_close'
                    ]
                ]
            ]
        ];
        
        $target = $this->getUserById($user_id);
        $mention = $this->getUserMention($target);
        $rank = $this->getUserRank($user_id);
        
        $text = "👤 <b>پنل مدیریت کاربر</b>\n\n";
        $text .= "👨‍💼 <b>کاربر:</b> {$mention}\n";
        $text .= "🆔 <b>آیدی:</b> <code>{$user_id}</code>\n";
        $text .= "👑 <b>مقام فعلی:</b> {$rank}\n\n";
        $text .= "━━━━━━━━━━━━━━━━\n\n";
        $text .= "🔹 <b>دسترسی‌های فعلی:</b>\n";
        $text .= "• بن: " . ($permissions['can_ban'] ? "✅ فعال" : "❌ غیرفعال") . "\n";
        $text .= "• سکوت: " . ($permissions['can_mute'] ? "✅ فعال" : "❌ غیرفعال") . "\n";
        $text .= "• حذف پیام: " . ($permissions['can_delete'] ? "✅ فعال" : "❌ غیرفعال") . "\n";
        $text .= "• سنجاق: " . ($permissions['can_pin'] ? "✅ فعال" : "❌ غیرفعال") . "\n\n";
        $text .= "💡 <b>راهنما:</b>\n";
        $text .= "• کلیک روی هر دکمه → تغییر وضعیت\n";
        $text .= "• ✅ = فعال | ❌ = غیرفعال\n";
        $text .= "• برای اعمال تغییرات روی «تایید و ذخیره» کلیک کنید";
        
        editMessage($this->chat_id, $message_id, $text, $keyboard);
    }
    
    /**
     * دریافت اطلاعات کاربر هدف - با 3 روش
     * پشتیبانی: ریپلای، @username، username (بدون @)، آیدی عددی
     */
    private function getTargetUser() {
        // روش 1: ریپلای
        if (isset($this->message['reply_to_message'])) {
            $target = $this->message['reply_to_message']['from'];
            $this->saveUserInfo($target);
            return [
                'id' => $target['id'],
                'name' => $target['first_name'] ?? 'کاربر',
                'username' => $target['username'] ?? null
            ];
        }
        
        // روش 2: یوزرنیم (با یا بدون @)
        // الگوی جدید: @username یا username (بدون @)
        if (preg_match('/@?(\w+)/', $this->text, $matches)) {
            $username = $matches[1]; // یوزرنیم بدون @
            
            // جستجو در دیتابیس
            $stmt = $this->db->prepare("SELECT user_id, first_name, username FROM mir_users WHERE username = ? AND group_id = ? ORDER BY last_seen DESC LIMIT 1");
            $stmt->execute([$username, $this->chat_id]);
            $user = $stmt->fetch();
            
            if ($user) {
                return [
                    'id' => $user['user_id'],
                    'name' => $user['first_name'] ?? 'کاربر',
                    'username' => $user['username']
                ];
            }
        }
        
        // روش 3: آیدی عددی (5 رقم یا بیشتر)
        if (preg_match('/\d{5,}/', $this->text, $matches)) {
            $user_id = $matches[0];
            
            // جستجو در دیتابیس
            $stmt = $this->db->prepare("SELECT user_id, first_name, username FROM mir_users WHERE user_id = ? AND group_id = ? ORDER BY last_seen DESC LIMIT 1");
            $stmt->execute([$user_id, $this->chat_id]);
            $user = $stmt->fetch();
            
            if ($user) {
                return [
                    'id' => $user['user_id'],
                    'name' => $user['first_name'] ?? 'کاربر',
                    'username' => $user['username']
                ];
            }
            
            // اگر در دیتابیس نبود، همان آیدی را برگردان
            return [
                'id' => $user_id,
                'name' => 'کاربر',
                'username' => null
            ];
        }
        
        return null;
    }
    
    /**
     * دریافت اطلاعات کاربر با آیدی
     */
    private function getUserById($user_id) {
        $stmt = $this->db->prepare("SELECT user_id, first_name as name, username FROM mir_users WHERE user_id = ? AND group_id = ? ORDER BY last_seen DESC LIMIT 1");
        $stmt->execute([$user_id, $this->chat_id]);
        $user = $stmt->fetch();
        
        if ($user) {
            return [
                'id' => $user['user_id'],
                'name' => $user['name'] ?? 'کاربر',
                'username' => $user['username']
            ];
        }
        
        return [
            'id' => $user_id,
            'name' => 'کاربر',
            'username' => null
        ];
    }
    
    /**
     * ساخت منشن کاربر
     */
    private function getUserMention($user) {
        $name = $user['name'];
        $user_id = $user['id'];
        return "<a href='tg://user?id={$user_id}'>{$name}</a>";
    }
    
    /**
     * بررسی سطح دسترسی
     */
    private function checkPermission($required_level) {
        if ($this->from_id == SUDO_ID) return true;
        if ($this->isCreator($this->from_id)) return true;
        
        if ($required_level == 'owner' || $required_level == 'deputy' || $required_level == 'advisor' || $required_level == 'admin' || $required_level == 'vip') {
            if ($this->isOwner($this->from_id)) return true;
        }
        
        if ($required_level == 'deputy' || $required_level == 'advisor' || $required_level == 'admin' || $required_level == 'vip') {
            if ($this->isDeputy($this->from_id)) return true;
        }
        
        if ($required_level == 'advisor' || $required_level == 'admin' || $required_level == 'vip') {
            if ($this->isAdvisor($this->from_id)) return true;
        }
        
        if ($required_level == 'admin' || $required_level == 'vip') {
            if ($this->isAdmin($this->from_id)) return true;
        }
        
        return false;
    }
    
    /**
     * بررسی دسترسی به پنل
     */
    private function checkPanelPermission($user_id = null) {
        $check_id = $user_id ?? $this->from_id;
        
        if ($check_id == SUDO_ID) return true;
        if ($this->isCreator($check_id)) return true;
        if ($this->isOwner($check_id)) return true;
        if ($this->isDeputy($check_id)) return true;
        
        return false;
    }
    
    /**
     * تنظیم مالک (وزیر اعظم)
     */
    private function setOwner() {
        if (!$this->checkPermission('owner')) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if ($target['id'] == SUDO_ID) {
            $this->reply("⛔️ نمی‌توانید سودو را تغییر دهید!");
            return true;
        }
        
        if ($this->isCreator($target['id'])) {
            $this->reply("⛔️ سازنده گروه نیاز به ارتقا ندارد!");
            return true;
        }
        
        if ($this->isOwner($target['id'])) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› در لیست مالکین وجود دارد!");
            return true;
        }
        
        $this->removeFromAllRanks($target['id']);
        
        $stmt = $this->db->prepare("INSERT INTO mir_owners (group_id, user_id, user_name, username, added_by, added_at) VALUES (?, ?, ?, ?, ?, NOW())");
        $stmt->execute([$this->chat_id, $target['id'], $target['name'], $target['username'], $this->from_id]);
        
        $mention = $this->getUserMention($target);
        $this->reply("✅ کاربر {$mention}\n›› به مقام مالک (وزیر اعظم) ارتقا یافت.");
        
        return true;
    }
    
    /**
     * حذف مالک
     */
    private function removeOwner() {
        if (!$this->checkPermission('owner')) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if (!$this->isOwner($target['id'])) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› در لیست مالکین نیست!");
            return true;
        }
        
        $stmt = $this->db->prepare("DELETE FROM mir_owners WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $target['id']]);
        
        $mention = $this->getUserMention($target);
        $this->reply("✅ کاربر {$mention}\n›› از مقام مالک عزل شد.");
        
        return true;
    }
    
    /**
     * لیست مالکین
     */
    private function listOwners() {
        $stmt = $this->db->prepare("SELECT * FROM mir_owners WHERE group_id = ? ORDER BY added_at DESC");
        $stmt->execute([$this->chat_id]);
        $owners = $stmt->fetchAll();
        
        if (empty($owners)) {
            $this->reply("📋 لیست مالکین خالی است.");
            return true;
        }
        
        $text = "👑 <b>لیست مالکین گروه</b>\n\n";
        $count = 1;
        
        foreach ($owners as $owner) {
            $mention = "<a href='tg://user?id={$owner['user_id']}'>{$owner['user_name']}</a>";
            $username = $owner['username'] ? "@{$owner['username']}" : "بدون یوزرنیم";
            $text .= "{$count}. {$mention}\n";
            $text .= "   › آیدی: <code>{$owner['user_id']}</code>\n";
            $text .= "   › یوزرنیم: {$username}\n\n";
            $count++;
        }
        
        $text .= "📊 تعداد: " . count($owners) . " نفر";
        
        $this->reply($text);
        return true;
    }
    
    /**
     * تنظیم معاون
     */
    private function setDeputy() {
        if (!$this->checkPermission('owner')) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if ($target['id'] == SUDO_ID) {
            $this->reply("⛔️ نمی‌توانید سودو را تغییر دهید!");
            return true;
        }
        
        if ($this->isCreator($target['id']) || $this->isOwner($target['id'])) {
            $this->reply("⚠️ این کاربر در مقام بالاتر است!");
            return true;
        }
        
        if ($this->isDeputy($target['id'])) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› در لیست معاونین وجود دارد!");
            return true;
        }
        
        $this->removeFromAllRanks($target['id']);
        
        $stmt = $this->db->prepare("INSERT INTO mir_deputies (group_id, user_id, user_name, username, added_by, added_at) VALUES (?, ?, ?, ?, ?, NOW())");
        $stmt->execute([$this->chat_id, $target['id'], $target['name'], $target['username'], $this->from_id]);
        
        $mention = $this->getUserMention($target);
        $this->reply("✅ کاربر {$mention}\n›› به مقام معاون ارتقا یافت.");
        
        return true;
    }
    
    /**
     * حذف معاون
     */
    private function removeDeputy() {
        if (!$this->checkPermission('owner')) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if (!$this->isDeputy($target['id'])) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› در لیست معاونین نیست!");
            return true;
        }
        
        $stmt = $this->db->prepare("DELETE FROM mir_deputies WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $target['id']]);
        
        $mention = $this->getUserMention($target);
        $this->reply("✅ کاربر {$mention}\n›› از مقام معاون عزل شد.");
        
        return true;
    }
    
    /**
     * لیست معاونین
     */
    private function listDeputies() {
        $stmt = $this->db->prepare("SELECT * FROM mir_deputies WHERE group_id = ? ORDER BY added_at DESC");
        $stmt->execute([$this->chat_id]);
        $deputies = $stmt->fetchAll();
        
        if (empty($deputies)) {
            $this->reply("📋 لیست معاونین خالی است.");
            return true;
        }
        
        $text = "🎖 <b>لیست معاونین گروه</b>\n\n";
        $count = 1;
        
        foreach ($deputies as $deputy) {
            $mention = "<a href='tg://user?id={$deputy['user_id']}'>{$deputy['user_name']}</a>";
            $username = $deputy['username'] ? "@{$deputy['username']}" : "بدون یوزرنیم";
            $text .= "{$count}. {$mention}\n";
            $text .= "   › آیدی: <code>{$deputy['user_id']}</code>\n";
            $text .= "   › یوزرنیم: {$username}\n\n";
            $count++;
        }
        
        $text .= "📊 تعداد: " . count($deputies) . " نفر";
        
        $this->reply($text);
        return true;
    }
    
    /**
     * تنظیم مشاور
     */
    private function setAdvisor() {
        if (!$this->checkPermission('deputy')) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if ($this->isProtected($target['id'])) {
            $this->reply("⚠️ این کاربر در مقام بالاتر است!");
            return true;
        }
        
        if ($this->isAdvisor($target['id'])) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› در لیست مشاورین وجود دارد!");
            return true;
        }
        
        $this->removeFromAllRanks($target['id']);
        
        $stmt = $this->db->prepare("INSERT INTO mir_advisors (group_id, user_id, user_name, username, added_by, added_at) VALUES (?, ?, ?, ?, ?, NOW())");
        $stmt->execute([$this->chat_id, $target['id'], $target['name'], $target['username'], $this->from_id]);
        
        $mention = $this->getUserMention($target);
        $this->reply("✅ کاربر {$mention}\n›› به مقام مشاور ارتقا یافت.");
        
        return true;
    }
    
    /**
     * حذف مشاور
     */
    private function removeAdvisor() {
        if (!$this->checkPermission('deputy')) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if (!$this->isAdvisor($target['id'])) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› در لیست مشاورین نیست!");
            return true;
        }
        
        $stmt = $this->db->prepare("DELETE FROM mir_advisors WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $target['id']]);
        
        $mention = $this->getUserMention($target);
        $this->reply("✅ کاربر {$mention}\n›› از مقام مشاور عزل شد.");
        
        return true;
    }
    
    /**
     * لیست مشاورین
     */
    private function listAdvisors() {
        $stmt = $this->db->prepare("SELECT * FROM mir_advisors WHERE group_id = ? ORDER BY added_at DESC");
        $stmt->execute([$this->chat_id]);
        $advisors = $stmt->fetchAll();
        
        if (empty($advisors)) {
            $this->reply("📋 لیست مشاورین خالی است.");
            return true;
        }
        
        $text = "📋 <b>لیست مشاورین گروه</b>\n\n";
        $count = 1;
        
        foreach ($advisors as $advisor) {
            $mention = "<a href='tg://user?id={$advisor['user_id']}'>{$advisor['user_name']}</a>";
            $username = $advisor['username'] ? "@{$advisor['username']}" : "بدون یوزرنیم";
            $text .= "{$count}. {$mention}\n";
            $text .= "   › آیدی: <code>{$advisor['user_id']}</code>\n";
            $text .= "   › یوزرنیم: {$username}\n\n";
            $count++;
        }
        
        $text .= "📊 تعداد: " . count($advisors) . " نفر";
        
        $this->reply($text);
        return true;
    }
    
    /**
     * تنظیم مدیر
     */
    private function setAdmin() {
        if (!$this->checkPermission('advisor')) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if ($this->isProtected($target['id']) && !$this->isAdmin($target['id']) && !$this->isVip($target['id'])) {
            $this->reply("⚠️ این کاربر در مقام بالاتر است!");
            return true;
        }
        
        if ($this->isAdmin($target['id'])) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› در لیست مدیران وجود دارد!");
            return true;
        }
        
        $this->removeFromAllRanks($target['id']);
        
        $stmt = $this->db->prepare("INSERT INTO mir_admins (group_id, user_id, user_name, username, added_by, added_at) VALUES (?, ?, ?, ?, ?, NOW())");
        $stmt->execute([$this->chat_id, $target['id'], $target['name'], $target['username'], $this->from_id]);
        
        $mention = $this->getUserMention($target);
        $this->reply("✅ کاربر {$mention}\n›› به مقام مدیر ارتقا یافت.");
        
        return true;
    }
    
    /**
     * حذف مدیر
     */
    private function removeAdmin() {
        if (!$this->checkPermission('advisor')) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if (!$this->isAdmin($target['id'])) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› در لیست مدیران نیست!");
            return true;
        }
        
        $stmt = $this->db->prepare("DELETE FROM mir_admins WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $target['id']]);
        
        $mention = $this->getUserMention($target);
        $this->reply("✅ کاربر {$mention}\n›› از مقام مدیر عزل شد.");
        
        return true;
    }
    
    /**
     * لیست مدیران
     */
    private function listAdmins() {
        $stmt = $this->db->prepare("SELECT * FROM mir_admins WHERE group_id = ? ORDER BY added_at DESC");
        $stmt->execute([$this->chat_id]);
        $admins = $stmt->fetchAll();
        
        if (empty($admins)) {
            $this->reply("📋 لیست مدیران خالی است.");
            return true;
        }
        
        $text = "👨‍💼 <b>لیست مدیران گروه</b>\n\n";
        $count = 1;
        
        foreach ($admins as $admin) {
            $mention = "<a href='tg://user?id={$admin['user_id']}'>{$admin['user_name']}</a>";
            $username = $admin['username'] ? "@{$admin['username']}" : "بدون یوزرنیم";
            $text .= "{$count}. {$mention}\n";
            $text .= "   › آیدی: <code>{$admin['user_id']}</code>\n";
            $text .= "   › یوزرنیم: {$username}\n\n";
            $count++;
        }
        
        $text .= "📊 تعداد: " . count($admins) . " نفر";
        
        $this->reply($text);
        return true;
    }
    
    /**
     * تنظیم ویژه
     */
    private function setVip() {
        if (!$this->checkPermission('admin')) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if ($this->isVip($target['id'])) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› در لیست ویژه وجود دارد!");
            return true;
        }
        
        $this->removeFromAllRanks($target['id']);
        
        $stmt = $this->db->prepare("INSERT INTO mir_vips (group_id, user_id, user_name, username, added_by, added_at) VALUES (?, ?, ?, ?, ?, NOW())");
        $stmt->execute([$this->chat_id, $target['id'], $target['name'], $target['username'], $this->from_id]);
        
        $mention = $this->getUserMention($target);
        $this->reply("✅ کاربر {$mention}\n›› به لیست ویژه اضافه شد.");
        
        return true;
    }
    
    /**
     * حذف ویژه
     */
    private function removeVip() {
        if (!$this->checkPermission('admin')) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if (!$this->isVip($target['id'])) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› در لیست ویژه نیست!");
            return true;
        }
        
        $stmt = $this->db->prepare("DELETE FROM mir_vips WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $target['id']]);
        
        $mention = $this->getUserMention($target);
        $this->reply("✅ کاربر {$mention}\n›› از لیست ویژه حذف شد.");
        
        return true;
    }
    
    /**
     * لیست ویژه
     */
    private function listVips() {
        $stmt = $this->db->prepare("SELECT * FROM mir_vips WHERE group_id = ? ORDER BY added_at DESC");
        $stmt->execute([$this->chat_id]);
        $vips = $stmt->fetchAll();
        
        if (empty($vips)) {
            $this->reply("📋 لیست ویژه خالی است.");
            return true;
        }
        
        $text = "⭐️ <b>لیست کاربران ویژه</b>\n\n";
        $count = 1;
        
        foreach ($vips as $vip) {
            $mention = "<a href='tg://user?id={$vip['user_id']}'>{$vip['user_name']}</a>";
            $username = $vip['username'] ? "@{$vip['username']}" : "بدون یوزرنیم";
            $text .= "{$count}. {$mention}\n";
            $text .= "   › آیدی: <code>{$vip['user_id']}</code>\n";
            $text .= "   › یوزرنیم: {$username}\n\n";
            $count++;
        }
        
        $text .= "📊 تعداد: " . count($vips) . " نفر";
        
        $this->reply($text);
        return true;
    }
    
    /**
     * بن کاربر - با 3 روش
     */
    private function banUser() {
        // بررسی دسترسی از پنل
        $permissions = $this->getUserPermissions($this->from_id);
        if (!$this->checkPermission('admin') && !$permissions['can_ban']) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if ($this->isProtected($target['id'])) {
            $this->reply("⛔️ نمی‌توانید مقام‌داران را بن کنید!");
            return true;
        }
        
        if ($this->isBanned($target['id'])) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› در لیست بن وجود دارد!");
            return true;
        }
        
        $stmt = $this->db->prepare("INSERT INTO mir_banned (group_id, user_id, user_name, username, banned_by, banned_at) VALUES (?, ?, ?, ?, ?, NOW())");
        $stmt->execute([$this->chat_id, $target['id'], $target['name'], $target['username'], $this->from_id]);
        
        sendRequest('banChatMember', [
            'chat_id' => $this->chat_id,
            'user_id' => $target['id']
        ]);
        
        $mention = $this->getUserMention($target);
        $this->reply("🚫 کاربر {$mention}\n›› از گروه بن شد.");
        
        return true;
    }
    
    /**
     * حذف بن - با 3 روش
     */
    private function unbanUser() {
        $permissions = $this->getUserPermissions($this->from_id);
        if (!$this->checkPermission('admin') && !$permissions['can_ban']) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if (!$this->isBanned($target['id'])) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› در لیست بن نیست!");
            return true;
        }
        
        $stmt = $this->db->prepare("DELETE FROM mir_banned WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $target['id']]);
        
        sendRequest('unbanChatMember', [
            'chat_id' => $this->chat_id,
            'user_id' => $target['id']
        ]);
        
        $mention = $this->getUserMention($target);
        $this->reply("✅ کاربر {$mention}\n›› از لیست بن خارج شد.");
        
        return true;
    }
    
    /**
     * لیست بن شدگان
     */
    private function listBanned() {
        $stmt = $this->db->prepare("SELECT * FROM mir_banned WHERE group_id = ? ORDER BY banned_at DESC");
        $stmt->execute([$this->chat_id]);
        $banned = $stmt->fetchAll();
        
        if (empty($banned)) {
            $this->reply("📋 لیست بن خالی است.");
            return true;
        }
        
        $text = "🚫 <b>لیست کاربران بن شده</b>\n\n";
        $count = 1;
        
        foreach ($banned as $ban) {
            $mention = "<a href='tg://user?id={$ban['user_id']}'>{$ban['user_name']}</a>";
            $username = $ban['username'] ? "@{$ban['username']}" : "بدون یوزرنیم";
            $text .= "{$count}. {$mention}\n";
            $text .= "   › آیدی: <code>{$ban['user_id']}</code>\n";
            $text .= "   › یوزرنیم: {$username}\n\n";
            $count++;
        }
        
        $text .= "📊 تعداد: " . count($banned) . " نفر";
        
        $this->reply($text);
        return true;
    }
    
    /**
     * بن سراسری - فقط سودو - بدون توجه به مقام
     */
    private function globalBan() {
        if ($this->from_id != SUDO_ID) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if ($target['id'] == SUDO_ID) {
            $this->reply("⛔️ نمی‌توانید خودتان را بن کنید!");
            return true;
        }
        
        // بررسی بن سراسری قبلی
        $stmt = $this->db->prepare("SELECT * FROM mir_global_banned WHERE user_id = ?");
        $stmt->execute([$target['id']]);
        if ($stmt->fetch()) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› از قبل در لیست بن سراسری است!");
            return true;
        }
        
        // افزودن به بن سراسری (بدون توجه به مقام)
        $stmt = $this->db->prepare("INSERT INTO mir_global_banned (user_id, user_name, username, banned_by, banned_at) VALUES (?, ?, ?, ?, NOW())");
        $stmt->execute([$target['id'], $target['name'], $target['username'], $this->from_id]);
        
        // بن از تمام گروه‌های فعال (بدون چک مقام)
        $stmt = $this->db->prepare("SELECT group_id FROM groups_data WHERE is_active = 1");
        $stmt->execute();
        $groups = $stmt->fetchAll();
        
        $banned_count = 0;
        foreach ($groups as $group) {
            $result = sendRequest('banChatMember', [
                'chat_id' => $group['group_id'],
                'user_id' => $target['id']
            ]);
            if ($result && $result['ok']) {
                $banned_count++;
            }
        }
        
        $mention = $this->getUserMention($target);
        $this->reply("🌐 <b>بن سراسری انجام شد!</b>\n\n" . 
                    "کاربر {$mention} از تمام گروه‌ها بن شد.\n\n" . 
                    "📊 تعداد گروه‌ها: {$banned_count}\n\n" . 
                    "⚠️ <b>توجه:</b> اگر این کاربر به صورت دستی آزاد شود، با اولین پیام خود دوباره خودکار بن می‌شود.");
        
        return true;
    }
    
    /**
     * حذف بن سراسری - آزادسازی واقعی
     */
    private function removeGlobalBan() {
        if ($this->from_id != SUDO_ID) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        // بررسی وجود در لیست بن سراسری
        $stmt = $this->db->prepare("SELECT * FROM mir_global_banned WHERE user_id = ?");
        $stmt->execute([$target['id']]);
        if (!$stmt->fetch()) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› در لیست بن سراسری نیست!");
            return true;
        }
        
        // حذف واقعی از لیست بن سراسری
        $stmt = $this->db->prepare("DELETE FROM mir_global_banned WHERE user_id = ?");
        $stmt->execute([$target['id']]);
        
        $mention = $this->getUserMention($target);
        $this->reply("✅ <b>آزادسازی کامل انجام شد!</b>\n\n" . 
                    "کاربر {$mention} از لیست بن سراسری خارج شد.\n\n" . 
                    "✅ این کاربر حالا می‌تواند آزادانه در گروه‌ها فعالیت کند.");
        
        return true;
    }
    
    /**
     * سکوت کاربر - با 3 روش
     */
    private function muteUser() {
        $permissions = $this->getUserPermissions($this->from_id);
        if (!$this->checkPermission('admin') && !$permissions['can_mute']) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if ($this->isProtected($target['id'])) {
            $this->reply("⛔️ نمی‌توانید مقام‌داران را سکوت کنید!");
            return true;
        }
        
        if ($this->isMuted($target['id'])) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› در لیست سکوت وجود دارد!");
            return true;
        }
        
        $stmt = $this->db->prepare("INSERT INTO mir_muted (group_id, user_id, user_name, username, muted_by, muted_at) VALUES (?, ?, ?, ?, ?, NOW())");
        $stmt->execute([$this->chat_id, $target['id'], $target['name'], $target['username'], $this->from_id]);
        
        sendRequest('restrictChatMember', [
            'chat_id' => $this->chat_id,
            'user_id' => $target['id'],
            'permissions' => json_encode([
                'can_send_messages' => false
            ])
        ]);
        
        $mention = $this->getUserMention($target);
        $this->reply("🔇 کاربر {$mention}\n›› سکوت شد.");
        
        return true;
    }
    
    /**
     * حذف سکوت - با 3 روش
     */
    private function unmuteUser() {
        $permissions = $this->getUserPermissions($this->from_id);
        if (!$this->checkPermission('admin') && !$permissions['can_mute']) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if (!$this->isMuted($target['id'])) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› در لیست سکوت نیست!");
            return true;
        }
        
        $stmt = $this->db->prepare("DELETE FROM mir_muted WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $target['id']]);
        
        sendRequest('restrictChatMember', [
            'chat_id' => $this->chat_id,
            'user_id' => $target['id'],
            'permissions' => json_encode([
                'can_send_messages' => true,
                'can_send_media_messages' => true,
                'can_send_other_messages' => true,
                'can_add_web_page_previews' => true
            ])
        ]);
        
        $mention = $this->getUserMention($target);
        $this->reply("✅ کاربر {$mention}\n›› از حالت سکوت خارج شد.");
        
        return true;
    }
    
    /**
     * لیست سکوت شدگان
     */
    private function listMuted() {
        $stmt = $this->db->prepare("SELECT * FROM mir_muted WHERE group_id = ? ORDER BY muted_at DESC");
        $stmt->execute([$this->chat_id]);
        $muted = $stmt->fetchAll();
        
        if (empty($muted)) {
            $this->reply("📋 لیست سکوت خالی است.");
            return true;
        }
        
        $text = "🔇 <b>لیست کاربران سکوت شده</b>\n\n";
        $count = 1;
        
        foreach ($muted as $mute) {
            $mention = "<a href='tg://user?id={$mute['user_id']}'>{$mute['user_name']}</a>";
            $username = $mute['username'] ? "@{$mute['username']}" : "بدون یوزرنیم";
            $text .= "{$count}. {$mention}\n";
            $text .= "   › آیدی: <code>{$mute['user_id']}</code>\n";
            $text .= "   › یوزرنیم: {$username}\n\n";
            $count++;
        }
        
        $text .= "📊 تعداد: " . count($muted) . " نفر";
        
        $this->reply($text);
        return true;
    }
    
    /**
     * اخطار به کاربر - با 3 روش
     */
    private function warnUser() {
        if (!$this->checkPermission('admin')) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        if ($this->isProtected($target['id'])) {
            $this->reply("⛔️ نمی‌توانید به مقام‌داران اخطار دهید!");
            return true;
        }
        
        $warn_limit = $this->getWarnLimit();
        
        $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM mir_warns WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $target['id']]);
        $current_warns = $stmt->fetch()['count'];
        
        $stmt = $this->db->prepare("INSERT INTO mir_warns (group_id, user_id, user_name, username, warned_by, warned_at) VALUES (?, ?, ?, ?, ?, NOW())");
        $stmt->execute([$this->chat_id, $target['id'], $target['name'], $target['username'], $this->from_id]);
        
        $current_warns++;
        
        $mention = $this->getUserMention($target);
        $remaining = $warn_limit - $current_warns;
        
        if ($current_warns >= $warn_limit) {
            $stmt = $this->db->prepare("INSERT IGNORE INTO mir_muted (group_id, user_id, user_name, username, muted_by, muted_at) VALUES (?, ?, ?, ?, ?, NOW())");
            $stmt->execute([$this->chat_id, $target['id'], $target['name'], $target['username'], 0]);
            
            sendRequest('restrictChatMember', [
                'chat_id' => $this->chat_id,
                'user_id' => $target['id'],
                'permissions' => json_encode(['can_send_messages' => false])
            ]);
            
            $this->reply("⚠️ کاربر {$mention}\n›› اخطار {$current_warns} از {$warn_limit} دریافت کرد.\n\n🔇 به دلیل رسیدن به حد مجاز، خودکار سکوت شد!");
        } else {
            $this->reply("⚠️ کاربر {$mention}\n›› اخطار {$current_warns} از {$warn_limit} دریافت کرد.\n\n📊 اخطار باقیمانده: {$remaining}");
        }
        
        return true;
    }
    
    /**
     * حذف یک اخطار - با 3 روش
     */
    private function removeWarn() {
        if (!$this->checkPermission('admin')) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        $stmt = $this->db->prepare("SELECT * FROM mir_warns WHERE group_id = ? AND user_id = ? ORDER BY warned_at DESC LIMIT 1");
        $stmt->execute([$this->chat_id, $target['id']]);
        $warn = $stmt->fetch();
        
        if (!$warn) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› اخطاری ندارد!");
            return true;
        }
        
        $stmt = $this->db->prepare("DELETE FROM mir_warns WHERE id = ?");
        $stmt->execute([$warn['id']]);
        
        $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM mir_warns WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $target['id']]);
        $remaining = $stmt->fetch()['count'];
        
        $mention = $this->getUserMention($target);
        $this->reply("✅ یک اخطار از کاربر {$mention} حذف شد.\n\n📊 اخطارهای باقیمانده: {$remaining}");
        
        return true;
    }
    
    /**
     * حذف تمام اخطارها - با 3 روش
     */
    private function clearWarns() {
        if (!$this->checkPermission('admin')) {
            return false;
        }
        
        $target = $this->getTargetUser();
        if (!$target) {
            $this->reply("❌ لطفاً روی پیام کاربر ریپلای بزنید یا یوزرنیم/آیدی را وارد کنید.");
            return true;
        }
        
        $stmt = $this->db->prepare("DELETE FROM mir_warns WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $target['id']]);
        $count = $stmt->rowCount();
        
        if ($count == 0) {
            $mention = $this->getUserMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› اخطاری ندارد!");
            return true;
        }
        
        $mention = $this->getUserMention($target);
        $this->reply("✅ تمام اخطارهای کاربر {$mention} حذف شد.\n\n📊 تعداد: {$count} اخطار");
        
        return true;
    }
    
    /**
     * تنظیم حد اخطار
     */
    private function setWarnLimit($limit) {
        if (!$this->checkPermission('owner')) {
            return false;
        }
        
        $limit = intval($limit);
        if ($limit < 1 || $limit > 20) {
            $this->reply("❌ حد اخطار باید بین 1 تا 20 باشد!");
            return true;
        }
        
        $stmt = $this->db->prepare("INSERT INTO mir_settings (group_id, setting_key, setting_value) VALUES (?, 'warn_limit', ?) ON DUPLICATE KEY UPDATE setting_value = ?");
        $stmt->execute([$this->chat_id, $limit, $limit]);
        
        $this->reply("✅ حد اخطار با موفقیت به {$limit} تنظیم شد.\n\n⚠️ پس از رسیدن به این تعداد، کاربر خودکار سکوت می‌شود.");
        
        return true;
    }
    
    /**
     * لیست اخطارها
     */
    private function listWarned() {
        $stmt = $this->db->prepare("
            SELECT user_id, user_name, username, COUNT(*) as warn_count 
            FROM mir_warns 
            WHERE group_id = ? 
            GROUP BY user_id, user_name, username
            ORDER BY warn_count DESC
        ");
        $stmt->execute([$this->chat_id]);
        $warned = $stmt->fetchAll();
        
        if (empty($warned)) {
            $this->reply("📋 لیست اخطارها خالی است.");
            return true;
        }
        
        $warn_limit = $this->getWarnLimit();
        
        $text = "⚠️ <b>لیست کاربران دارای اخطار</b>\n\n";
        $text .= "📊 حد مجاز: {$warn_limit} اخطار\n\n";
        $count = 1;
        
        foreach ($warned as $warn) {
            $mention = "<a href='tg://user?id={$warn['user_id']}'>{$warn['user_name']}</a>";
            $username = $warn['username'] ? "@{$warn['username']}" : "بدون یوزرنیم";
            $text .= "{$count}. {$mention}\n";
            $text .= "   › اخطارها: {$warn['warn_count']} از {$warn_limit}\n";
            $text .= "   › یوزرنیم: {$username}\n\n";
            $count++;
        }
        
        $text .= "📊 تعداد: " . count($warned) . " نفر";
        
        $this->reply($text);
        return true;
    }
    
    // توابع کمکی
    
    private function isGroupActive() {
        $stmt = $this->db->prepare("SELECT is_active FROM groups_data WHERE group_id = ?");
        $stmt->execute([$this->chat_id]);
        $result = $stmt->fetch();
        return $result && $result['is_active'] == 1;
    }
    
    private function isCreator($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM group_admins WHERE group_id = ? AND user_id = ? AND is_creator = 1");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isOwner($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_owners WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isDeputy($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_deputies WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isAdvisor($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_advisors WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isAdmin($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_admins WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isVip($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_vips WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isBanned($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_banned WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isMuted($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_muted WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isProtected($user_id) {
        if ($user_id == SUDO_ID) return true;
        if ($this->isCreator($user_id)) return true;
        if ($this->isOwner($user_id)) return true;
        if ($this->isDeputy($user_id)) return true;
        if ($this->isAdvisor($user_id)) return true;
        if ($this->isAdmin($user_id)) return true;
        if ($this->isVip($user_id)) return true;
        return false;
    }
    
    private function removeFromAllRanks($user_id) {
        $stmt = $this->db->prepare("DELETE FROM mir_owners WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        
        $stmt = $this->db->prepare("DELETE FROM mir_deputies WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        
        $stmt = $this->db->prepare("DELETE FROM mir_advisors WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        
        $stmt = $this->db->prepare("DELETE FROM mir_admins WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        
        $stmt = $this->db->prepare("DELETE FROM mir_vips WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
    }
    
    private function getUserRank($user_id) {
        if ($user_id == SUDO_ID) return "👨‍💻 سودو";
        if ($this->isCreator($user_id)) return "👑 سازنده گروه";
        if ($this->isOwner($user_id)) return "💎 وزیر اعظم";
        if ($this->isDeputy($user_id)) return "🎖 معاون";
        if ($this->isAdvisor($user_id)) return "📋 مشاور";
        if ($this->isAdmin($user_id)) return "👨‍💼 مدیر";
        if ($this->isVip($user_id)) return "⭐️ ویژه";
        return "👤 کاربر عادی";
    }
    
    private function getWarnLimit() {
        $stmt = $this->db->prepare("SELECT setting_value FROM mir_settings WHERE group_id = ? AND setting_key = 'warn_limit'");
        $stmt->execute([$this->chat_id]);
        $result = $stmt->fetch();
        return $result ? intval($result['setting_value']) : 5;
    }
    
    private function getUserPermissions($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_permissions WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        $perm = $stmt->fetch();
        
        if ($perm) {
            return [
                'can_ban' => (bool)$perm['can_ban'],
                'can_mute' => (bool)$perm['can_mute'],
                'can_delete' => (bool)$perm['can_delete'],
                'can_pin' => (bool)$perm['can_pin']
            ];
        }
        
        return [
            'can_ban' => false,
            'can_mute' => false,
            'can_delete' => false,
            'can_pin' => false
        ];
    }
    
    private function saveTempPermissions($user_id, $permissions) {
        $data = json_encode($permissions);
        $stmt = $this->db->prepare("INSERT INTO mir_temp_data (group_id, user_id, data_key, data_value) VALUES (?, ?, 'panel_permissions', ?) ON DUPLICATE KEY UPDATE data_value = ?");
        $stmt->execute([$this->chat_id, $user_id, $data, $data]);
    }
    
    private function getTempPermissions($user_id) {
        $stmt = $this->db->prepare("SELECT data_value FROM mir_temp_data WHERE group_id = ? AND user_id = ? AND data_key = 'panel_permissions'");
        $stmt->execute([$this->chat_id, $user_id]);
        $result = $stmt->fetch();
        
        if ($result) {
            return json_decode($result['data_value'], true);
        }
        
        return $this->getUserPermissions($user_id);
    }
    
    private function clearTempPermissions($user_id) {
        $stmt = $this->db->prepare("DELETE FROM mir_temp_data WHERE group_id = ? AND user_id = ? AND data_key = 'panel_permissions'");
        $stmt->execute([$this->chat_id, $user_id]);
    }
    
    private function saveUserPermissions($user_id, $permissions) {
        $stmt = $this->db->prepare("
            INSERT INTO mir_permissions (group_id, user_id, can_ban, can_mute, can_delete, can_pin, updated_at) 
            VALUES (?, ?, ?, ?, ?, ?, NOW()) 
            ON DUPLICATE KEY UPDATE 
            can_ban = ?, can_mute = ?, can_delete = ?, can_pin = ?, updated_at = NOW()
        ");
        $stmt->execute([
            $this->chat_id,
            $user_id,
            $permissions['can_ban'] ? 1 : 0,
            $permissions['can_mute'] ? 1 : 0,
            $permissions['can_delete'] ? 1 : 0,
            $permissions['can_pin'] ? 1 : 0,
            $permissions['can_ban'] ? 1 : 0,
            $permissions['can_mute'] ? 1 : 0,
            $permissions['can_delete'] ? 1 : 0,
            $permissions['can_pin'] ? 1 : 0
        ]);
    }
    
    private function updateUserRank($user_id, $permissions) {
        $this->removeFromAllRanks($user_id);
        
        $count = 0;
        if ($permissions['can_ban']) $count++;
        if ($permissions['can_mute']) $count++;
        if ($permissions['can_delete']) $count++;
        if ($permissions['can_pin']) $count++;
        
        if ($count == 4) {
            $stmt = $this->db->prepare("INSERT INTO mir_admins (group_id, user_id, user_name, username, added_by, added_at) VALUES (?, ?, ?, ?, ?, NOW())");
            $stmt->execute([$this->chat_id, $user_id, 'کاربر', null, $this->from_id]);
        } elseif ($count > 0 && $count < 4) {
            $stmt = $this->db->prepare("INSERT INTO mir_advisors (group_id, user_id, user_name, username, added_by, added_at) VALUES (?, ?, ?, ?, ?, NOW())");
            $stmt->execute([$this->chat_id, $user_id, 'کاربر', null, $this->from_id]);
        }
    }
    
    private function reply($text) {
        sendMessage($this->chat_id, $text, null, $this->message_id);
    }
    
    public function saveUserInfo($user) {
        $stmt = $this->db->prepare("
            INSERT INTO mir_users (group_id, user_id, first_name, username, last_seen) 
            VALUES (?, ?, ?, ?, NOW()) 
            ON DUPLICATE KEY UPDATE 
            first_name = ?, username = ?, last_seen = NOW()
        ");
        $stmt->execute([
            $this->chat_id,
            $user['id'],
            $user['first_name'] ?? 'کاربر',
            $user['username'] ?? null,
            $user['first_name'] ?? 'کاربر',
            $user['username'] ?? null
        ]);
    }
    
    /**
     * بررسی بن سراسری - بن خودکار با اولین پیام
     */
    public function checkGlobalBan($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_global_banned WHERE user_id = ?");
        $stmt->execute([$user_id]);
        $banned = $stmt->fetch();
        
        if ($banned) {
            // بن خودکار (حتی اگر به صورت دستی آزاد شده باشد)
            $result = sendRequest('banChatMember', [
                'chat_id' => $this->chat_id,
                'user_id' => $user_id
            ]);
            
            // اطلاع به سودو
            if ($result && $result['ok']) {
                $notify = "🚨 <b>بن خودکار انجام شد</b>\n\n";
                $notify .= "👤 <b>کاربر:</b> <a href='tg://user?id={$user_id}'>{$banned['user_name']}</a>\n";
                $notify .= "🆔 <b>آیدی:</b> <code>{$user_id}</code>\n";
                $notify .= "👥 <b>گروه:</b> {$this->chat_id}\n\n";
                $notify .= "⚠️ این کاربر در لیست بن سراسری است و به صورت خودکار دوباره بن شد.";
                
                sendMessage(SUDO_ID, $notify);
            }
            
            return true;
        }
        
        return false;
    }
}

function processMirCommand($message) {
    $mir = new MirSystem($message);
    // اطلاعات کاربر قبلاً در handleGroupMessage ذخیره شده
    // checkGlobalBan هم قبلاً انجام شده
    $mir->processCommand();
}

function handleMirCallback($callback) {
    $data = $callback['data'];
    
    if (strpos($data, 'panel_') === 0) {
        $message = $callback['message'];
        $mir = new MirSystem($message);
        $mir->handlePanelCallback(
            $data,
            $callback['id'],
            $message['message_id'],
            $callback['from']['id']
        );
        return true;
    }
    
    return false;
}

?>