<?php
/**
 * ═══════════════════════════════════════════════════════
 * 🔐 سیستم نجوای پیشرفته (Whisper Pro) - نسخه مدرن
 * ═══════════════════════════════════════════════════════
 * ✅ رفع باگ متن‌های طولانی (ارسال خودکار به PV)
 * ✅ طراحی بصری مینیمال و شیک
 * ✅ سیستم آمارگیر دقیق
 * ✅ امنیت کامل حریم خصوصی
 */

require_once 'config.php';
require_once 'jdate.php';

/**
 * 🛠 ایجاد جداول دیتابیس (در صورت عدم وجود)
 */
function createWhisperTables(): void
{
    $db = Database::instance()->pdo();
    
    // جدول اصلی نجواها
    $db->exec("
        CREATE TABLE IF NOT EXISTS whispers (
            id INT AUTO_INCREMENT PRIMARY KEY,
            token VARCHAR(64) UNIQUE NOT NULL,
            group_id BIGINT NOT NULL,
            message_id INT NOT NULL,
            sender_id BIGINT NOT NULL,
            sender_name VARCHAR(255) NOT NULL,
            receiver_id BIGINT NOT NULL,
            receiver_name VARCHAR(255) NOT NULL,
            whisper_text TEXT DEFAULT NULL,
            is_read TINYINT(1) DEFAULT 0,
            is_expired TINYINT(1) DEFAULT 0,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_token (token)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    
    // جدول آمار بازدید
    $db->exec("
        CREATE TABLE IF NOT EXISTS whisper_views (
            id INT AUTO_INCREMENT PRIMARY KEY,
            token VARCHAR(64) NOT NULL,
            user_id BIGINT NOT NULL,
            user_name VARCHAR(255) NOT NULL,
            viewed_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY unique_view (token, user_id),
            INDEX idx_token (token)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    
    // جدول وضعیت کاربران (State)
    $db->exec("
        CREATE TABLE IF NOT EXISTS user_states (
            user_id BIGINT PRIMARY KEY,
            step VARCHAR(50) DEFAULT NULL,
            whisper_token VARCHAR(64) DEFAULT NULL,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
}
createWhisperTables();

/**
 * 📨 پردازش دستور نجوا در گروه (!نجوا یا نجوا)
 */
function processWhisperCommand(array $message): bool
{
    $text = trim($message['text'] ?? '');
    $chatId = $message['chat']['id'];
    $sender = $message['from'];
    
    if ($text !== 'نجوا') {
        return false;
    }
    
    // بررسی ریپلای
    if (!isset($message['reply_to_message'])) {
        sendMessage($chatId, "⚠️ <b>خطا در ارسال:</b>\nبرای ارسال پیام محرمانه، باید روی پیام شخص مورد نظر <b>ریپلای</b> کنید.", null, $message['message_id']);
        return true;
    }
    
    $receiver = $message['reply_to_message']['from'];
    
    // اعتبارسنجی‌ها
    if ($sender['id'] === $receiver['id']) {
        sendMessage($chatId, "❌ <b>ارسال ناموفق:</b> نمی‌توانید برای خودتان نجوا بفرستید.", null, $message['message_id']);
        return true;
    }
    if (isset($receiver['is_bot']) && $receiver['is_bot']) {
        sendMessage($chatId, "🤖 <b>ربات‌ها راز نگه دار نیستند!</b>\nامکان ارسال نجوا به ربات وجود ندارد.", null, $message['message_id']);
        return true;
    }
    
    // تولید توکن و لینک عمیق
    $token = bin2hex(random_bytes(16));
    $botUsername = str_replace('@', '', BOT_USERNAME);
    $deepLink = "https://t.me/{$botUsername}?start=w_{$token}";
    
    $senderName = htmlspecialchars($sender['first_name']);
    $receiverName = htmlspecialchars($receiver['first_name']);
    
    // متن پیام شیک اولیه
    $txt = "🔐 <b>درخواست ارسال محرمانه</b>\n\n";
    $txt .= "👤 <b>فرستنده:</b> " . getUserMention($sender) . "\n";
    $txt .= "🎯 <b>گیرنده:</b> " . getUserMention($receiver) . "\n\n";
    $txt .= "✍️ <b>وضعیت:</b> در انتظار نگارش متن...";
    
    $keyboard = [
        'inline_keyboard' => [
            [['text' => '📝 ثبت متن محرمانه', 'url' => $deepLink]]
        ]
    ];
    
    $sent = sendMessage($chatId, $txt, $keyboard, $message['message_id']);
    
    // ذخیره در دیتابیس
    if ($sent && isset($sent['result']['message_id'])) {
        $db = Database::instance()->pdo();
        $stmt = $db->prepare("
            INSERT INTO whispers (token, group_id, message_id, sender_id, sender_name, receiver_id, receiver_name)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $token,
            $chatId,
            $sent['result']['message_id'],
            $sender['id'],
            $senderName,
            $receiver['id'],
            $receiverName
        ]);
    }
    
    return true;
}

/**
 * 🏃‍♂️ هندلر استارت در PV (دریافت توکن و درخواست متن)
 */
function handleWhisperStart(array $message, string $param): void
{
    $token = str_replace('w_', '', $param);
    $userId = $message['from']['id'];
    $chatId = $message['chat']['id'];
    
    $db = Database::instance()->pdo();
    $stmt = $db->prepare("SELECT * FROM whispers WHERE token = ? LIMIT 1");
    $stmt->execute([$token]);
    $whisper = $stmt->fetch();
    
    if (!$whisper) {
        sendMessage($chatId, "⚠️ <b>خطا:</b> این لینک نامعتبر یا منقضی شده است.");
        return;
    }
    
    if ($whisper['sender_id'] != $userId) {
        sendMessage($chatId, "⛔️ <b>دسترسی غیرمجاز:</b> شما فرستنده این نجوا نیستید.");
        return;
    }
    
    if ($whisper['whisper_text'] !== null) {
        sendMessage($chatId, "✅ <b>متن ثبت شده:</b> پیام این نجوا قبلاً ارسال شده است.");
        return;
    }
    
    // تنظیم وضعیت کاربر
    $stmt = $db->prepare("
        INSERT INTO user_states (user_id, step, whisper_token) VALUES (?, 'awaiting_whisper', ?)
        ON DUPLICATE KEY UPDATE step = 'awaiting_whisper', whisper_token = ?
    ");
    $stmt->execute([$userId, $token, $token]);
    
    $txt = "📝 <b>متن پیام محرمانه را بنویسید:</b>\n\n";
    $txt .= "👤 گیرنده: <b>" . htmlspecialchars($whisper['receiver_name']) . "</b>\n\n";
    $txt .= "💡 <i>نکته: متن می‌تواند طولانی باشد و شامل استیکر یا ایموجی شود.</i>\n\n";
    $txt .= "❌ لغو عملیات: <code>/cancel</code>";
    
    sendMessage($chatId, $txt);
}

/**
 * 📥 دریافت و ذخیره متن نجوا در PV
 */
function receiveWhisperText(array $message): bool
{
    $userId = $message['from']['id'];
    $text = $message['text'] ?? '';
    
    if (empty($text)) return false;
    
    $db = Database::instance()->pdo();
    $stmt = $db->prepare("SELECT * FROM user_states WHERE user_id = ? AND step = 'awaiting_whisper'");
    $stmt->execute([$userId]);
    $state = $stmt->fetch();
    
    if (!$state) return false;
    
    // انصراف
    if ($text === '/cancel') {
        $db->prepare("DELETE FROM user_states WHERE user_id = ?")->execute([$userId]);
        sendMessage($userId, "🚫 عملیات لغو شد.");
        return true;
    }
    
    $token = $state['whisper_token'];
    
    // ذخیره متن
    $stmt = $db->prepare("UPDATE whispers SET whisper_text = ? WHERE token = ?");
    $stmt->execute([$text, $token]);
    
    // پاک کردن وضعیت
    $db->prepare("DELETE FROM user_states WHERE user_id = ?")->execute([$userId]);
    
    // دریافت اطلاعات برای آپدیت
    $stmt = $db->prepare("SELECT * FROM whispers WHERE token = ?");
    $stmt->execute([$token]);
    $whisper = $stmt->fetch();
    
    sendMessage($userId, "✅ <b>عالی!</b> پیام شما رمزنگاری و در گروه ارسال شد.");
    
    // بروزرسانی پیام در گروه
    updateGroupWhisperMessage($whisper);
    
    return true;
}

/**
 * 🔄 بروزرسانی پیام گروه (UI/UX اصلی)
 */
function updateGroupWhisperMessage($whisper)
{
    $isRead = $whisper['is_read'];
    $isExpired = $whisper['is_expired'];
    
    $senderMention = "<a href='tg://user?id={$whisper['sender_id']}'>" . htmlspecialchars($whisper['sender_name']) . "</a>";
    $receiverMention = "<a href='tg://user?id={$whisper['receiver_id']}'>" . htmlspecialchars($whisper['receiver_name']) . "</a>";
    
    // 🔴 حالت منقضی شده
    if ($isExpired) {
        $txt = "🔒 <b>پیام محرمانه (منقضی شده)</b>\n\n";
        $txt .= "📤 <b>فرستنده:</b> {$senderMention}\n";
        $txt .= "📥 <b>گیرنده:</b> {$receiverMention}\n\n";
        $txt .= "🗑 <i>این پیام توسط فرستنده حذف شد و دیگر قابل خواندن نیست.</i>";
        
        $keyboard = [
            'inline_keyboard' => [
                [['text' => '📊 آمار بازدید', 'callback_data' => "stats_whisper_{$whisper['token']}"]]
            ]
        ];
    }
    // 🟢 حالت خوانده شده
    elseif ($isRead) {
        $txt = "📂 <b>پیام محرمانه (خوانده شده)</b>\n\n";
        $txt .= "📤 <b>فرستنده:</b> {$senderMention}\n";
        $txt .= "📥 <b>گیرنده:</b> {$receiverMention}\n\n";
        $txt .= "✅ وضعیت: <b>توسط گیرنده مشاهده شد</b>";
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '👁 مشاهده مجدد', 'callback_data' => "show_whisper_{$whisper['token']}"],
                    ['text' => '🗑 ابطال پیام', 'callback_data' => "delete_whisper_{$whisper['token']}"]
                ]
            ]
        ];
    }
    // 🔵 حالت جدید (ارسال شده)
    else {
        $txt = "📬 <b>پیام محرمانه جدید</b>\n\n";
        $txt .= "📤 <b>فرستنده:</b> {$senderMention}\n";
        $txt .= "📥 <b>گیرنده:</b> {$receiverMention}\n\n";
        $txt .= "🔔 <i>محتوای پیام فقط برای گیرنده قابل نمایش است.</i>";
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '👁 مشاهده پیام', 'callback_data' => "show_whisper_{$whisper['token']}"],
                    ['text' => '↩️ پاسخ سریع', 'callback_data' => "reply_whisper_{$whisper['token']}"]
                ]
            ]
        ];
    }
    
    editMessage($whisper['group_id'], $whisper['message_id'], $txt, $keyboard);
}

/**
 * 🔘 مدیریت دکمه‌های شیشه‌ای (نسخه فیکس شده برای متون طولانی)
 */
function handleWhisperCallback(array $callback): bool
{
    $data = $callback['data'];
    $userId = $callback['from']['id'];
    $userName = $callback['from']['first_name'];
    $callbackId = $callback['id'];
    
    // 👁️ دکمه نمایش
    if (strpos($data, 'show_whisper_') === 0) {
        $token = str_replace('show_whisper_', '', $data);
        
        $db = Database::instance()->pdo();
        $stmt = $db->prepare("SELECT * FROM whispers WHERE token = ? LIMIT 1");
        $stmt->execute([$token]);
        $whisper = $stmt->fetch();
        
        if (!$whisper) {
            answerCallbackQuery($callbackId, "❌ پیام یافت نشد!", true);
            return true;
        }
        
        // ثبت آمار
        logWhisperView($token, $userId, $userName);
        
        // چک انقضا
        if ($whisper['is_expired']) {
            answerCallbackQuery($callbackId, "⛔️ این پیام منقضی شده است.", true);
            return true;
        }
        
        // چک دسترسی
        if ($userId != $whisper['sender_id'] && $userId != $whisper['receiver_id']) {
            answerCallbackQuery($callbackId, "⛔️ این پیام محرمانه برای شما نیست! 🤫", true);
            return true;
        }
        
        // --- 🚀 بخش مهم: مدیریت متن‌های طولانی ---
        $fullText = $whisper['whisper_text'];
        
        // محدودیت تلگرام حدود 200 کاراکتر برای Alert است. ما 190 در نظر می‌گیریم.
        if (mb_strlen($fullText) > 190) {
            // ارسال به PV
            $pvMsg = "💬 <b>متن نجوای دریافتی:</b>\n\n" . $fullText;
            $sendResult = sendMessage($userId, $pvMsg);
            
            if (isset($sendResult['ok']) && $sendResult['ok']) {
                answerCallbackQuery($callbackId, "✅ متن طولانی بود و به پی‌وی (PV) شما ارسال شد. چک کنید!", false);
            } else {
                answerCallbackQuery($callbackId, "❌ متن طولانی است اما ربات نمی‌تواند به شما پیام دهد. لطفا ربات را استارت کنید.", true);
            }
        } else {
            // نمایش پاپ‌آپ (روش استاندارد)
            answerCallbackQuery($callbackId, "💬 متن پیام:\n\n" . $fullText, true);
        }
        // ------------------------------------------
        
        // تغییر وضعیت به خوانده شده
        if ($userId == $whisper['receiver_id'] && !$whisper['is_read']) {
            $stmt = $db->prepare("UPDATE whispers SET is_read = 1 WHERE token = ?");
            $stmt->execute([$token]);
            
            $whisper['is_read'] = 1;
            updateGroupWhisperMessage($whisper);
        }
        
        return true;
    }
    
    // 🗑 دکمه حذف
    if (strpos($data, 'delete_whisper_') === 0) {
        $token = str_replace('delete_whisper_', '', $data);
        $db = Database::instance()->pdo();
        $stmt = $db->prepare("SELECT * FROM whispers WHERE token = ?");
        $stmt->execute([$token]);
        $whisper = $stmt->fetch();
        
        if (!$whisper) return true;
        
        if ($userId != $whisper['sender_id']) {
            answerCallbackQuery($callbackId, "⛔️ فقط فرستنده می‌تواند پیام را باطل کند.", true);
            return true;
        }
        
        $stmt = $db->prepare("UPDATE whispers SET is_expired = 1 WHERE token = ?");
        $stmt->execute([$token]);
        
        $whisper['is_expired'] = 1;
        updateGroupWhisperMessage($whisper);
        
        answerCallbackQuery($callbackId, "✅ پیام با موفقیت باطل شد.");
        return true;
    }
    
    // 📊 دکمه آمار
    if (strpos($data, 'stats_whisper_') === 0) {
        $token = str_replace('stats_whisper_', '', $data);
        $db = Database::instance()->pdo();
        $stmt = $db->prepare("SELECT * FROM whispers WHERE token = ?");
        $stmt->execute([$token]);
        $whisper = $stmt->fetch();
        
        if ($userId != $whisper['sender_id']) {
            answerCallbackQuery($callbackId, "⛔️ آمار فقط برای فرستنده است.", true);
            return true;
        }
        
        $stmt = $db->prepare("SELECT user_name, COUNT(*) as count FROM whisper_views WHERE token = ? GROUP BY user_id, user_name");
        $stmt->execute([$token]);
        $views = $stmt->fetchAll();
        
        $statsText = "📊 <b>آمار کنجکاوی‌ها:</b>\n\n";
        $total = 0;
        
        if (empty($views)) {
            $statsText .= "▫️ هنوز کسی تلاش نکرده است.";
        } else {
            foreach ($views as $v) {
                $statsText .= "👤 <b>{$v['user_name']}</b>\n";
                $total++;
            }
        }
        
        // اگر متن آمار هم خیلی طولانی باشد، آن را هم کوتاه می‌کنیم
        if (mb_strlen($statsText) > 190) {
             answerCallbackQuery($callbackId, "📊 " . $total . " نفر دکمه را فشار دادند (لیست کامل در دیتابیس).", true);
        } else {
             answerCallbackQuery($callbackId, str_replace(['<b>', '</b>'], '', $statsText), true); // حذف تگ HTML برای Alert
        }
        
        return true;
    }
    
    // ↩️ دکمه پاسخ
    if (strpos($data, 'reply_whisper_') === 0) {
        $token = str_replace('reply_whisper_', '', $data);
        $db = Database::instance()->pdo();
        $stmt = $db->prepare("SELECT * FROM whispers WHERE token = ?");
        $stmt->execute([$token]);
        $whisper = $stmt->fetch();
        
        if ($userId != $whisper['receiver_id']) {
            answerCallbackQuery($callbackId, "⛔️ شما مخاطب این پیام نیستید.", true);
            return true;
        }
        
        answerCallbackQuery($callbackId, "↩️ برای پاسخ، روی پیام ریپلای کنید و بنویسید: نجوا", true);
        return true;
    }
    
    return false;
}

/**
 * 📝 ثبت لاگ بازدید
 */
function logWhisperView($token, $userId, $userName) {
    $db = Database::instance()->pdo();
    $stmt = $db->prepare("
        INSERT IGNORE INTO whisper_views (token, user_id, user_name, viewed_at) 
        VALUES (?, ?, ?, NOW())
    ");
    $stmt->execute([$token, $userId, $userName]);
}
?>
