<?php
if (basename($_SERVER['PHP_SELF']) === basename(__FILE__)) {
    http_response_code(403);
    die('Direct access not allowed');
}

date_default_timezone_set('Asia/Tehran');
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/time.php';

function log_debug($message) {
    $logFile = __DIR__ . '/debug.log';
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) @mkdir($logDir, 0755, true);
    if (!is_writable($logDir)) return;
    $timestamp = date('Y-m-d H:i:s');
    @file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

function db() {
    static $pdo;
    if ($pdo) return $pdo;
    try {
        $pdo = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4", DB_USER, DB_PASS, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false
        ]);
    } catch (PDOException $e) {
        log_debug("DB Error: " . $e->getMessage());
        http_response_code(500);
        die("DB Error");
    }
    return $pdo;
}

function sendTelegram($method, $data = []) {
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/$method";
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($data, JSON_UNESCAPED_UNICODE),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_TIMEOUT => 10,
        CURLOPT_CONNECTTIMEOUT => 5,
        CURLOPT_SSL_VERIFYPEER => true
    ]);
    $res = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    if (curl_errno($ch) || $http_code !== 200) {
        if ($http_code !== 400) {
            log_debug("cURL Error [$method]: " . curl_error($ch) . " | HTTP: $http_code");
        }
        curl_close($ch);
        return false;
    }
    curl_close($ch);
    $decoded = json_decode($res, true);
    if (!($decoded['ok'] ?? false)) {
        $error_code = $decoded['error_code'] ?? 0;
        if (!in_array($error_code, [400, 403, 429])) {
            log_debug("API Error [$method]: " . ($decoded['description'] ?? 'Unknown'));
        }
    }
    return $decoded;
}

function setCache($key, $value, $ttl = 3600) {
    $expire = date('Y-m-d H:i:s', time() + $ttl);
    try {
        db()->prepare("REPLACE INTO cache (k, v, expire_at) VALUES (?, ?, ?)")->execute([$key, $value, $expire]);
    } catch (Exception $e) {}
}

function getCache($key) {
    try {
        $stmt = db()->prepare("SELECT v FROM cache WHERE k = ? AND expire_at > NOW()");
        $stmt->execute([$key]);
        return $stmt->fetchColumn();
    } catch (Exception $e) {
        return null;
    }
}

function hasProcessedUpdate(string $type, int $id): bool {
    $key = "processed_update_{$type}_{$id}";
    if (getCache($key)) return true;
    setCache($key, '1', 60);
    return false;
}

function getUserRole($chat_id, $user_id) {
    if ((int)$user_id === OWNER_ID) return 'owner_bot';
    $owner_id = getGroupOwnerFromChat($chat_id);
    if ($owner_id && (int)$user_id === (int)$owner_id) return 'owner_group';
    
    $stmt = db()->prepare("SELECT 1 FROM group_owners WHERE chat_id = ? AND user_id = ?");
    $stmt->execute([$chat_id, $user_id]);
    if ($stmt->fetch()) return 'owner_assigned';
    
    $stmt = db()->prepare("SELECT 1 FROM vip_users WHERE chat_id = ? AND user_id = ?");
    $stmt->execute([$chat_id, $user_id]);
    if ($stmt->fetch()) return 'vip';
    
    $stmt = db()->prepare("SELECT can_ban, can_mute, can_delete, can_pin, can_vip FROM bot_admins WHERE chat_id = ? AND user_id = ?");
    $stmt->execute([$chat_id, $user_id]);
    if ($row = $stmt->fetch()) {
        return [
            'role' => 'admin',
            'can_ban' => (bool)$row['can_ban'],
            'can_mute' => (bool)($row['can_mute'] ?? false),
            'can_delete' => (bool)($row['can_delete'] ?? false),
            'can_pin' => (bool)($row['can_pin'] ?? false),
            'can_vip' => (bool)($row['can_vip'] ?? false)
        ];
    }
    return 'member';
}

// فقط مالک ربات و سازنده گروه
function isGroupOwner(int $chat_id, int $user_id): bool {
    return in_array(getUserRole($chat_id, $user_id), ['owner_bot', 'owner_group']);
}

function canManageLocks(int $chat_id, int $user_id): bool {
    $role = getUserRole($chat_id, $user_id);
    return isGroupOwner($chat_id, $user_id) || 
           (is_array($role) && $role['role'] === 'admin' && ($role['can_vip'] ?? false));
}

function canPinMessages(int $chat_id, int $user_id): bool {
    $role = getUserRole($chat_id, $user_id);
    return isGroupOwner($chat_id, $user_id) || 
           (is_array($role) && $role['role'] === 'admin' && ($role['can_pin'] ?? false));
}

function canBanUsers(int $chat_id, int $user_id): bool {
    $role = getUserRole($chat_id, $user_id);
    return isGroupOwner($chat_id, $user_id) || 
           (is_array($role) && $role['role'] === 'admin' && ($role['can_ban'] ?? false));
}

function isUserBannable($chat_id, $user_id): bool {
    $role = getUserRole($chat_id, $user_id);
    // مالک ربات، سازنده، مالک اختصاصی، مدیر، ویژه — همه غیرقابل بن
    return !in_array($role, ['owner_bot', 'owner_group', 'owner_assigned', 'vip']) && 
           (!is_array($role) || $role['role'] !== 'admin');
}

function setLock($chat_id, $lock, $state) {
    db()->prepare("INSERT INTO group_locks (chat_id, lock_type, is_locked) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE is_locked = ?")
        ->execute([$chat_id, $lock, (int)$state, (int)$state]);
    $key = "locks_{$chat_id}";
    $current = getCache($key);
    $locks = $current ? json_decode($current, true) : [];
    $locks[$lock] = (bool)$state;
    setCache($key, json_encode($locks), 86400 * 7);
}

function isLocked($chat_id, $lock) {
    $key = "locks_{$chat_id}";
    if ($data = getCache($key)) {
        $locks = json_decode($data, true);
        return !empty($locks[$lock]);
    }
    $stmt = db()->prepare("SELECT is_locked FROM group_locks WHERE chat_id = ? AND lock_type = ?");
    $stmt->execute([$chat_id, $lock]);
    $is_locked = (bool)$stmt->fetchColumn();
    $rs = db()->prepare("SELECT lock_type, is_locked FROM group_locks WHERE chat_id = ?");
    $rs->execute([$chat_id]);
    $all_locks = [];
    while ($row = $rs->fetch()) {
        $all_locks[$row['lock_type']] = (bool)$row['is_locked'];
    }
    setCache($key, json_encode($all_locks), 86400 * 7);
    return $is_locked;
}

function cacheUser($user_id, $first_name, $last_name = null, $username = null) {
    try {
        db()->prepare("INSERT INTO user_cache (user_id, first_name, last_name, username) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE first_name=VALUES(first_name),last_name=VALUES(last_name),username=VALUES(username)")->execute([$user_id, $first_name, $last_name, $username]);
    } catch (Exception $e) {}
}

function getFullUserName($user_id) {
    $stmt = db()->prepare("SELECT first_name, last_name, username FROM user_cache WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch();
    if (!$user) return "کاربر ($user_id)";
    $name = $user['first_name'];
    if ($user['last_name']) $name .= ' ' . $user['last_name'];
    return $name;
}

function formatUserMention($user_id) {
    $name = htmlspecialchars(getFullUserName($user_id), ENT_QUOTES, 'UTF-8');
    return "<a href='tg://user?id=$user_id'>$name</a>";
}

function getGroupOwnerFromChat($chat_id) {
    $cacheKey = "owner_chat_{$chat_id}";
    if ($cached = getCache($cacheKey)) return (int)$cached;
    $res = sendTelegram('getChatAdministrators', ['chat_id' => $chat_id]);
    if (!$res || !($res['ok'] ?? false) || empty($res['result'])) return null;
    foreach ($res['result'] as $admin) {
        if (($admin['status'] ?? '') === 'creator' && isset($admin['user']['id'])) {
            $ownerId = (int)$admin['user']['id'];
            setCache($cacheKey, $ownerId, 3600);
            return $ownerId;
        }
    }
    return null;
}

function isGroupActive($chat_id) {
    $stmt = db()->prepare("SELECT 1 FROM groups WHERE chat_id = ? AND expires_at > NOW()");
    $stmt->execute([$chat_id]);
    return (bool)$stmt->fetch();
}

function sendExpiryWarningIfNeeded($chat_id) {
    $stmt = db()->prepare("SELECT owner_id, expires_at FROM groups WHERE chat_id = ? AND expires_at BETWEEN NOW() AND DATE_ADD(NOW(), INTERVAL 24 HOUR) AND (last_expiry_warning IS NULL OR last_expiry_warning < DATE_SUB(NOW(), INTERVAL 23 HOUR))");
    $stmt->execute([$chat_id]);
    $row = $stmt->fetch();
    if ($row) {
        sendTelegram('sendMessage', [
            'chat_id' => $row['owner_id'],
            'text' => "⚠️ توجه!\nاعتبار ربات گروه شما ظرف 24 ساعت آینده به پایان می‌رسد.",
            'parse_mode' => 'HTML'
        ]);
        db()->prepare("UPDATE groups SET last_expiry_warning = NOW() WHERE chat_id = ?")->execute([$chat_id]);
    }
}

function scheduleMessageDeletion($chat_id, $message_id, $delay_seconds = 60) {
    $key = "delete_msg_{$chat_id}_{$message_id}";
    setCache($key, json_encode(['chat_id' => $chat_id, 'message_id' => $message_id]), $delay_seconds + 10);
}

function sendMessageWithAutoDelete($chat_id, $text, $reply_to = null, $delay = 60, $parse_mode = 'HTML') {
    $params = ['chat_id' => $chat_id, 'text' => $text, 'parse_mode' => $parse_mode];
    if ($reply_to) $params['reply_to_message_id'] = $reply_to;
    $result = sendTelegram('sendMessage', $params);
    if ($result && isset($result['result']['message_id'])) {
        scheduleMessageDeletion($chat_id, $result['result']['message_id'], $delay);
    }
    return $result;
}